/*
   Double Mode Frequency Meter / Jan 2026 / https://tol.acritum.com/2110/attiny814-frequency-meter
   Added:
    * Support for attiny814 and 128x64 OLED screen.
    * Tiny4kOLED output.
    * Interval mode measurements (1Hz - 25kHz) + addon RPM measurement.
    * Addon period time measurement for the original FREQ mode.
    * Rising/Falling edge, divider/multiplier, screen inverse settings.
    * Settings are saved to EEPROM and restored after power off.
    * RTC millis(). Set "millis()/micors() timer: Disabled".

   Based on the following work:

   100MHz Frequency Meter - see http://www.technoblogy.com/show?20B4
   David Johnson-Davies - www.technoblogy.com - 16th March 2021
   ATtiny414 = 20 MHz (internal oscillator; BOD disabled)

   CC BY 4.0
   Licensed under a Creative Commons Attribution 4.0 International license:
   http://creativecommons.org/licenses/by/4.0/
*/

/*
 Enter settings menu:
  Option 1 - device powered off: hold button, apply power, keep holding until menu appears.
  Option 2 - device running: hold button for 3+ seconds, release, press again and hold until menu appears (without holding - just reboot).
 Navigation in menu:
  If no button is pressed for 3 seconds, cursor moves to next setting item.
  If button is pressed, current setting value cycles and timeout resets to 3 seconds.
  After all settings are adjusted, values are saved to EEPROM and device reboots.
*/

// you can further compress font if memory is low !

//#define TINY4KOLED_QUICK_BEGIN   // this is for white oled only. for blue oled should be removed!

#include <Tiny4kOLED.h>
#include <avr/eeprom.h>
#include <font6x8_tol_minimal.h>

#define INTERVAL_MAX_TRIES 3 // how many additional seconds to wait for signal in interval mode (only for low frequencies <=1 Hz; for others always 1 second)
#define BUTTON_DENOISE 300 // debounce delay for button. my button is glitchy, usually 30-50 is enough.
#define FONT_NAME FONT6X8TOL

typedef struct {
    uint8_t starting_byte;
    uint8_t mode;
    uint8_t edge;
    uint8_t modifier;
    uint8_t screen;
} config_t;

config_t saved; // for fast EEPROM write

uint8_t mode; // 0 - Freq mode, 1 - Int mode.

volatile uint32_t rtc_overflow_count;
uint32_t rtc_last_update; // for screen update once per second

// for freq mode
volatile uint16_t ovf_count; // number of TCD0 overflows
volatile uint32_t counter; // total TCD0 ticks
volatile bool frequency_ready;

// for interval mode
volatile uint32_t pulse_count;
volatile uint32_t signal_starttime;
volatile uint32_t signal_endtime;
uint8_t interval_try; // if no signal, we will increase wait time

volatile bool button_pressed = false;
uint32_t button_last_pressed; // time of last button press
volatile uint8_t button_reset_request;

uint8_t saved_isc_edge; // stores selected edge

// for menu
uint8_t menu_state = 0;
uint16_t menu_timeout = 0;

inline void reset() {
    CCP = 0xD8;
    RSTCTRL.SWRR = 1;
}

// RTC interrupt handler
ISR(RTC_CNT_vect) {
    RTC.INTFLAGS = RTC_OVF_bm; // Clear interrupt flag
    rtc_overflow_count++;

    //PORTA.OUT ^= PIN4_bm;                             // Toggle LED
}

// Replacement for millis(), works for 49.7 days before overflow
uint32_t rtc_millis() {
    uint16_t cnt;
    uint32_t overflow;

    cli();
    cnt = RTC.CNT;
    overflow = rtc_overflow_count;

    // Check for missed overflow 
    // If OVF flag is set and cnt is low, overflow occurred but ISR not yet processed
    if ((RTC.INTFLAGS & RTC_OVF_bm) && cnt < 16384) overflow++;
    sei();


    // return (overflow * 1000UL) + (cnt * 1000UL / 32768UL); // 12949 миллисекунд на миллион операций
    // return ((uint64_t)overflow << 15 | cnt) * 1000 / 32768;  // 32306 миллисекунд на миллион операций
    //  return ((uint32_t)overflow * 1000UL) +  (cnt - (cnt >> 5) + (cnt >> 7) );   //11769 миллисекунд на миллион операций не работает
    //  return  uint32( (overflow << 10) - (overflow << 4) - (overflow << 3) +  (cnt - (cnt >> 5) + (cnt >> 7) )  ;  // 11734 миллисекунд на миллион операций не работает

    return ((uint32_t)overflow * 1000UL) + ((uint32_t)cnt * 125UL >> 12); // 11996 ms per million ops
}

void rtc_delay(uint16_t time) {
    uint16_t start = rtc_millis();
    while ((rtc_millis() - start) <= time);
}

// PORTA interrupt handler
ISR(PORTA_PORT_vect) {
    if (mode == 1 && (PORTA.INTFLAGS & PIN1_bm)) {
        PORTA.INTFLAGS = PIN1_bm; // Clear interrupt flag

        uint16_t cnt;
        uint32_t overflow;

        cnt = RTC.CNT;
        overflow = rtc_overflow_count;

        if (RTC.INTFLAGS & RTC_OVF_bm) {
            if (cnt < 16384) { // 16384 is half of 32768
                overflow += 1;
            }
        }
        //32 bit: to be used with millis - fast but low precision!
        //uint32_t now_time = ((uint32_t)overflow * 1000UL) + ((uint32_t)cnt * 125UL >> 12);  
        
        // 64 bit: optimal precision, but slow
        uint32_t now_time = ((uint64_t)overflow << 15 | cnt) * 1000000UL / 32768UL;

        if (pulse_count == 0) signal_starttime = now_time;
        else signal_endtime = now_time;

        pulse_count++;
    } // PIN1 - signal

    if (PORTA.INTFLAGS & PIN5_bm) { // button
        PORTA.INTFLAGS = PIN5_bm; // Clear interrupt flag

        // Button pressed (LOW)
        if (!(PORTA.IN & PIN5_bm)) {
            button_pressed = true;
            //PORTA.OUT ^= PIN4_bm;
        }

        // Button released
        if ((PORTA.IN & PIN5_bm)) {
            if (rtc_millis() - button_last_pressed > 3000 && !button_pressed) {
                button_reset_request = true;
            }
        }
    } // PIN5 - button
}

// Timer/Counter TCD0 overflow interrupt counts MSByte
ISR(TCD0_OVF_vect) {
    TCD0.INTFLAGS = TCD_OVF_bm; // Clear overflow interrupt flag
    ovf_count++;
}

// Timer/Counter TCD0 capture interrupt
ISR(TCD0_TRIG_vect) {
    TCD0.INTFLAGS = TCD_TRIGB_bm; // Clear capture interrupt flag

    counter = TCD0.CAPTUREB;
    counter = (uint32_t)ovf_count << 12 | counter;
    ovf_count = 0;
    frequency_ready = true;
    TCD0.INTFLAGS = TCD_OVF_bm; // Clear overflow interrupt flag
}

void nicevalue(uint32_t num, char* buffer) {
    if (num == 0) {
        buffer[0] = '0';
        buffer[1] = '\0';
        return;
    }

    char temp[12]; // enough for 10 digits
    int i = 0;

    // Write digits in reverse order (least significant first)
    while (num) {
        temp[i++] = '0' + (num % 10);
        num /= 10;
    }

    // Write to buffer in correct order, inserting separators
    int j = 0;
    for (int k = 0; k < i; k++) {
        if (k > 0 && (i - k) % 3 == 0) {
            buffer[j++] = ';';
        }
        buffer[j++] = temp[i - 1 - k];
    }
    buffer[j] = '\0';
}

void nicevalue_interval(uint32_t num, char* buffer) {
    uint32_t whole = num / 100;      // integer part
    uint32_t frac  = num % 100;      // fractional part (0–99)

    // Format integer part
    nicevalue(whole, buffer);

    // Find end of string
    char* end = buffer;
    while (*end) end++;

    // Add decimal point
    *end++ = ':';

    // Add two fractional digits (with leading zero)
    *end++ = '0' + (char)(frac / 10);
    *end++ = '0' + (char)(frac % 10);
    *end   = '\0';
}

void oled_print_centered(const char* text, int y, bool x2) {
    uint8_t charwidth = 6;
    if (x2) {
        oled.setFontX2Smooth(FONT_NAME);
        charwidth = 12;
    } else {
        oled.setFont(FONT_NAME);
    }

    uint8_t width = 0;
    for (int i = 0; text[i]; i++) {
        width += charwidth;
    }

    uint8_t x = (128 - width) / 2;
    oled.setCursor(0, y);
    oled.clearToEOL();
    oled.setCursor(x, y);
    oled.print(text);
}

void setMode(int8_t setmode) {
    if (setmode == 0) { // freq mode
        PORTA.PIN1CTRL |= PORT_ISC_INTDISABLE_gc; // Disable PA1 interrupt
        PORTA.PIN1CTRL = PORT_INVEN_bm;           // Invert input (PA1)

        ovf_count = 0;
        frequency_ready = false;
        mode = 0;

        oled_print_centered("FRQ:=MODE==5HZ@100MHZ", 0, false);
    } else { // interval mode
        // This disables inversion which breaks PA1 interrupts;
        // you can take the signal from PA2, but it's easier to just disable inversion
        PORTA.PIN1CTRL &= ~PORT_INVEN_bm;

        // Re-enable PA1 interrupt with saved edge setting
        PORTA.PIN1CTRL |= saved_isc_edge;
        interval_try = 0;
        mode = 1;

        oled_print_centered("INT:=MODE==1HZ@25KHZ", 0, false);
    }
}

bool is_button_pressed() {
    if (button_pressed) {
        // Button debounce
        rtc_delay(BUTTON_DENOISE);
        button_pressed = false;
        return true;
    } else return false;
}

void show_menu() {
    static uint8_t last_values[4] = {255, 255, 255, 255};
    const uint8_t current_values[] = {saved.mode, saved.edge, saved.modifier, saved.screen};

    for (uint8_t i = 0; i < 4; i++) {
        oled.setCursor(90, i + 1);
        oled.print(i == menu_state - 1 ? ">" : "=");

        if (current_values[i] != last_values[i]) {
            oled.setCursor(100, i + 1);
            oled.print("===");
            oled.setCursor(100, i + 1);
            oled.print(current_values[i]);
            last_values[i] = current_values[i];
        }
    }
}

void handle_menu() {
    if (is_button_pressed()) {
        menu_timeout = rtc_millis() + 3000;

        if (menu_state == 0) {
            menu_state = 1;
        } else {
            switch (menu_state) {
                case 1: saved.mode = (saved.mode + 1) % 2; break; // Mode: 0, 1   //    saved.mode = (saved.mode % 2) + 1; break; // Mode: 1, 2
                case 2: saved.edge = (saved.edge + 1) % 2; break;  // Edge: 0, 1
                case 3: saved.modifier = (saved.modifier + 1) % 21; break; // Modifier: 0-20
                case 4:
                    saved.screen = (saved.screen + 1) % 2;
                    oled.setInverse(saved.screen);
                    break;
            }
        }
    }

    // Auto-navigation by timeout
    if (menu_state > 0 && rtc_millis() > menu_timeout) {
        if (++menu_state > 4) {
            // Exit menu with save
            menu_state = 0;
            oled.setCursor(10, 6);
            oled.print("SAVE");
            saved.starting_byte = 117;
            eeprom_update_block(&saved, (void*)0, sizeof(config_t));
            rtc_delay(1000);
            reset();
        } else {
            menu_timeout = rtc_millis() + 3000;
        }
    }

    // Display menu if active
    if (menu_state > 0) {
        show_menu();
    }
}

// Setup **********************************************
void setup() {
    PORTA.DIRSET = PIN4_bm;                           // Make LED on PA4 an output
    PORTA.DIRCLR = PIN5_bm;                           // Button input
    PORTA.PIN5CTRL = PORT_PULLUPEN_bm | PORT_ISC_BOTHEDGES_gc; // Pull-up, both edges

    /// TIMER SETUP
    TCD0.CTRLB = TCD_WGMODE_ONERAMP_gc;               // One ramp waveform mode
    TCD0.CMPBCLR = 0xFFF;                             // Count up to maximum
    TCD0.INPUTCTRLB = TCD_INPUTMODE_EDGETRIG_gc;      // Capture and reset counter
    TCD0.EVCTRLB = TCD_CFG_ASYNC_gc | TCD_ACTION_bm | TCD_TRIGEI_bm; // Enable event
    TCD0.INTCTRL = TCD_OVF_bm | TCD_TRIGB_bm;         // Enable interrupts

    // Ensure ENRDY bit is set
    while (!(TCD0.STATUS & TCD_ENRDY_bm));

    // External clock, no prescaler, enable timer
    TCD0.CTRLA = TCD_CLKSEL_EXTCLK_gc | TCD_CNTPRES_DIV1_gc | TCD_ENABLE_bm;

    /// EVENT SYSTEM SETUP
    EVSYS.ASYNCCH1 = EVSYS_ASYNCCH1_RTC_OVF_gc;       // RTC overflow → event
    EVSYS.ASYNCUSER7 = EVSYS_ASYNCUSER7_ASYNCCH1_gc;  // Event triggers TCD0 capture (TCD0_TRIG_vect)
    EVSYS.ASYNCCH0 = EVSYS_ASYNCCH0_PORTA_PIN1_gc;    // PA1 is event generator for ASYNCCH0
    EVSYS.ASYNCUSER8 = EVSYS_ASYNCUSER8_ASYNCCH0_gc;  // ASYNCUSER8 is EVOUT0 (PA2) connects to ASYNCCH0
    PORTMUX.CTRLA = PORTMUX_EVOUT0_bm;                // Enable EVOUT0 (PA2)

    /// RTC SETUP: Initialize 32.768kHz Oscillator
    CPU_CCP = CCP_IOREG_gc; // Enable writing to protected register
    CLKCTRL.XOSC32KCTRLA = CLKCTRL.XOSC32KCTRLA & ~CLKCTRL_ENABLE_bm; // Disable oscillator

    while (CLKCTRL.MCLKSTATUS & CLKCTRL_XOSC32KS_bm); // Wait until XOSC32KS is 0

    CPU_CCP = CCP_IOREG_gc; // Enable writing to protected register
    CLKCTRL.XOSC32KCTRLA = CLKCTRL.XOSC32KCTRLA & ~CLKCTRL_SEL_bm; // Use External Crystal

    CPU_CCP = CCP_IOREG_gc; // Enable writing to protected register
    CLKCTRL.XOSC32KCTRLA = CLKCTRL.XOSC32KCTRLA | CLKCTRL_ENABLE_bm; // Enable oscillator

    // Initialize RTC
    while (RTC.STATUS > 0);                           // Wait until registers synchronized
    RTC.PER = 32767;                                   // Set period = 1 second
    RTC.CLKSEL = RTC_CLKSEL_TOSC32K_gc;               // 32.768kHz External Crystal Oscillator
    RTC.CTRLA = RTC_PRESCALER_DIV1_gc | RTC_RTCEN_bm; // Prescaler /1, enable RTC
    RTC.INTCTRL = RTC_OVF_bm; // Enable overflow interrupt

    /// OLED SETUP
    oled.begin(128, 64, sizeof(tiny4koled_init_defaults), tiny4koled_init_defaults);
    oled.enableChargePump();

    // Some newer devices do not contain an external current reference.
    // Older devices may also support using the internal curret reference,
    // which provides more consistent brightness across devices.
    // The internal current reference can be configured as either low current, or high current.
    // Using true as the parameter value choses the high current internal current reference,
    // resulting in a brighter display, and a more effective contrast setting.

    oled.setInternalIref(true); // Use internal current reference for brightness
    oled.setRotation(1);
    //oled.disableZoomIn(); // use if the screen is doubled
    oled.clear();
    oled.on();

    // Load saved settings
    eeprom_read_block(&saved, (void*)0, sizeof(config_t));

    if (saved.starting_byte != 117) {
        saved.mode = 0;     // mode
        saved.edge = 0;     // 0 - rising, 1 - falling
        saved.modifier = 0; // 0 - disabled; 1-10 - divide 1..10; 11-20 - multiply 1..10
        saved.screen = 0;   // 1 - screen invert
    }

    if (saved.screen == 1) oled.setInverse(true);

    sei();

    // Check if button is pressed at startup → enter menu
    if (!(PORTA.IN & PIN5_bm)) {
        oled_print_centered("SETTINGS", 0, false);
        oled.setCursor(0, 1);
        oled.println("MODE=FRQ<INT");
        oled.println("EDGE=RISE<FALL");
        oled.println("DIV<MUL");
        oled.print("SCREEN=INV");

        show_menu();

        // Wait for button release
        while (!(PORTA.IN & PIN5_bm));

        // Enter menu loop
        while (1) handle_menu();
    } else {
        // Set edge detection based on saved setting
        if (saved.edge == 1) {
            saved_isc_edge = PORT_ISC_FALLING_gc;
            TCD0.EVCTRLB |= TCD_EDGE_FALL_LOW_gc;
        } else {
            saved_isc_edge = PORT_ISC_RISING_gc;
            TCD0.EVCTRLB |= TCD_EDGE_RISE_HIGH_gc;
        }

        setMode(saved.mode);

        oled.setCursor(0, 7);
        if (saved.edge == 0) {
            oled.print("RISING=EDGE");
        } else {
            oled.print("FALLING=EDGE");
        }

        if (saved.modifier > 1) {
            oled.setCursor(104, 7);
            if (saved.modifier < 11) {
                oled.print("?=");
                oled.print(saved.modifier);
            } else if (saved.modifier > 11) {
                oled.print("X=");
                oled.print(saved.modifier - 10);
            }
        }
    }

// benchmarks section
/*
uint32_t start, end;
start = rtc_millis();

for (uint32_t i =0; i<10000000; i++)
{
// do something
}

end = rtc_millis();

oled.setCursor(1, 3);
oled.print(end-start); 
while (1) ;

*/


}

void loop() {
    if (is_button_pressed()) {
        button_last_pressed = rtc_millis();

        if (button_reset_request) reset(); // got second press → reset

        // if no reset, the button acts as a switch
        setMode(!mode);
    }

    if (button_reset_request) {
        oled_print_centered("RST", 2, true);
        return; // wait for second press to reset
    }

    if (rtc_overflow_count <= rtc_last_update) return;
    rtc_last_update = rtc_overflow_count;

    // MODE 0: Frequency Measurement
    if (mode == 0) {
        if (!frequency_ready) rtc_delay(100); // Wait a bit if measurement delayed

        if (frequency_ready) {
            uint32_t f_freq;
            cli();
            f_freq = counter;
            frequency_ready = false; // Reset flag
            sei();

            // I have no idea why it counts -1 Hz at low frequences 8-/
            // This is a temporary shitty fix for it... remove it if not needed...
            if (f_freq < 65536) f_freq++;

            if (saved.modifier > 1) {
                if (saved.modifier < 11) f_freq = f_freq / saved.modifier;
                else if (saved.modifier > 11) f_freq = f_freq * (saved.modifier - 10);
            }

            char temp_str[16];
            nicevalue(f_freq, temp_str);
            oled_print_centered(temp_str, 2, true);

            uint32_t val;
            char temp_str1[4];
            temp_str1[0] = '=';
            temp_str1[2] = 'S';
            temp_str1[3] = '\0';

            if (f_freq > 1000000) {
                val = 1000000000;
                temp_str1[1] = 'N';
            } else if (f_freq > 1000) {
                val = 1000000;
                temp_str1[1] = 'U';
            } else if (f_freq > 1) {
                val = 1000;
                temp_str1[1] = 'M';
            }

            nicevalue(val / f_freq, temp_str);
            strcat(temp_str, temp_str1);
            oled_print_centered(temp_str, 5, false);
        } else {
            // No signal
            oled_print_centered("0", 2, true);
            oled_print_centered("NO=SIGNAL", 5, false);
        }
    } // MODE 0 END
    // MODE 1: Interval Measurement
    else if (mode == 1) {
        if (pulse_count < 2) {
            interval_try++;
            if (interval_try <= INTERVAL_MAX_TRIES) return;
        }
        interval_try = 0;

        cli();
        uint32_t pul_cnt = pulse_count;
        uint32_t sig_start = signal_starttime;
        uint32_t sig_end = signal_endtime;
        sei();

        uint32_t sig_per = sig_end - sig_start;

        if (pul_cnt > 1 && sig_per > 0) {   // got at least 2 pulses
            
            // to be used with millis - low preceision
            // uint64_t tempval = (uint64_t)(pul_cnt - 1) * 100000; 
          
            // high precision
            uint64_t tempval = (uint64_t)(pul_cnt - 1) * 100000000ULL;
            uint32_t i_freq = tempval / sig_per;

            if (saved.modifier > 1) {
                if (saved.modifier < 11) i_freq = i_freq / saved.modifier;
                else if (saved.modifier > 11) i_freq = i_freq * (saved.modifier - 10);
            }

            char temp_str[16];
            nicevalue_interval(i_freq, temp_str);
            oled_print_centered(temp_str, 2, true);

            nicevalue(i_freq * 6 / 10, temp_str);
            strcat(temp_str, "=PPM");
            oled_print_centered(temp_str, 5, false);

            cli();
            pulse_count = 0;
            sei();
        } else {
            // Waiting for signal
            oled_print_centered("0", 2, true);
            oled_print_centered("NO=SIGNAL", 5, false);
        }
    } //MODE1 END
}
